<?php

namespace App\Http\Controllers;

use App\Models\Proceso;
use App\Models\Parametro;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Barryvdh\Snappy\Facades\SnappyPdf;
use RealRashid\SweetAlert\Facades\Alert;


class ProcesoController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $procesos = Proceso::where('estado', true)->with('tipo_proceso')->orderBy('fecha_inicio', 'desc')->paginate(10);
        $estados_proceso=Parametro::where('tipo','ESTADO_PROCESO')->get();
        $title = 'Delete User!';
        $text = "Are you sure you want to delete?";
        confirmDelete($title, $text);
        return view('procesos.index', compact('procesos','estados_proceso'));
    }

    public function procesos_finalizados()
    {
        $procesos = Proceso::where('estado', false)->with('tipo_proceso')->orderBy('fecha_inicio', 'desc')->paginate(10);
        $title = 'Delete User!';
        $text = "Are you sure you want to delete?";
        confirmDelete($title, $text);
        return view('procesos.finalizados', compact('procesos'));
    }
    
    public function create()
    {
        $tipos_proceso = Parametro::where('tipo', 'AUDIENCIA')->get();
        return view('procesos.create', compact('tipos_proceso'));
    }

    public function datos_proceso(Request $request)
    {
        $proceso = Proceso::find($request->proceso_id);
        return response()->json($proceso);
    }
    
    public function store(Request $request)
    {
        Proceso::create($request->validate([
            'numero_nurej'=>'required|unique:procesos',
            'parte_demandante'=>'required',
            'nombre_proceso'=>'required',
            'tipo_proceso_id'=>'required|exists:parametros,id',
            'fecha_inicio'=>'required|date',
            'delito'=>'required'
        ]));

        Alert::success('Éxito', 'Proceso judicial creado exitosamente.');

        return redirect()->route('procesos.index');
    }

    /**
     * Display the specified resource.
     */
    public function show(Proceso $proceso)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Proceso $proceso)
    {
        $tipos_proceso = Parametro::where('tipo', 'AUDIENCIA')->get();
        return view('procesos.edit', compact('proceso', 'tipos_proceso'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Proceso $proceso)
    {
        $proceso->update($request->validate([
            'numero_nurej' => [
                'required',
                Rule::unique('procesos', 'numero_nurej')->ignore($proceso->id),
            ],
            'parte_demandante'=>'required',
            'nombre_proceso'=>'required',
            'tipo_proceso_id'=>'required|exists:parametros,id',
            'fecha_inicio'=>'required|date',
            'delito'=>'required'
        ]));
        Alert::success('Éxito', 'Proceso judicial actualizado exitosamente.');
        return redirect()->route('procesos.index');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Proceso $proceso)
    {
        $proceso->delete();
        Alert::success('Éxito', 'Proceso judicial eliminado exitosamente.');
        return redirect()->route('procesos.index');
    }

    public function pdf(Proceso $proceso)
    {
        $proceso->load(['seguimientos']);

        $pdf = SnappyPdf::loadView('procesos.pdf', compact('proceso'))
            ->setOption('margin-top', '15mm')
            ->setOption('margin-bottom', '15mm')
            ->setOption('margin-left', '15mm')
            ->setOption('margin-right', '15mm')
            ->setOption('encoding', 'UTF-8')
            ->setOption('enable-local-file-access', true);

        return $pdf->stream('proceso_'.$proceso->codigo_proceso.'.pdf');
    }
}
