<?php

namespace App\Http\Controllers\Auth;

use App\Models\User;
use LdapRecord\Container;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;

use Illuminate\Foundation\Auth\AuthenticatesUsers;
use LdapRecord\Models\ActiveDirectory\User as LdapUser;

class LoginController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Login Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles authenticating users for the application and
    | redirecting them to your home screen. The controller uses a trait
    | to conveniently provide its functionality to your applications.
    |
    */

    use AuthenticatesUsers;

    /**
     * Where to redirect users after login.
     *
     * @var string
     */
    protected $redirectTo = '/home';

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('guest')->except('logout');
        $this->middleware('auth')->only('logout');
    }

    public function username()
    {
        return 'username';
    }

    public function login(Request $request)
    {
        $request->validate([
            'username' => 'required|string',
            'password' => 'required|string',
            'captcha' => 'required|captcha',
        ], [
            'username.required' => 'Debe ingresar el nombre de usuario asignado',
            'password.required' => 'Debe ingresar su contraseña',
            'captcha.required' => 'Debe ingresar el código de verificación',
            'captcha.captcha' => 'El código de verificación es incorrecto o ha expirado. Por favor, refresca e inténtalo nuevamente.'         
        ]);

        $credentials = $request->only('username', 'password');

        $ldapUser = LdapUser::where('samaccountname', '=', $credentials['username'])->first();
        if ($ldapUser) {
            // Autenticación en LDAP
            if (Container::getConnection()->auth()->attempt($ldapUser->getDn(), $credentials['password'])) {
                $nombre_usuario = $ldapUser->getFirstAttribute('samaccountname');
                
				if(User::where('username', $nombre_usuario)->exists()){
					$usuario = User::where('username', $nombre_usuario)->first();
					if($usuario) {
						Auth::guard('web')->login($usuario);
						return redirect()->intended('home');
					} else {
						return back()->withErrors([
							'message' => 'Usuario no esta habilitado para acceder al sistema, comuniquese con el administrador del sistema.',
						]);
					}
				}
                else{
                    return back()->withErrors([
                        'message' => 'Usuario no existe en la base de datos, comuniquese con el administrador del sistema.',
                    ]);
                }
            }
            else{
                $attempt = Auth::attempt($credentials);
                if($attempt){
                    return redirect()->intended('home');
                }
                else{
                    return back()->withErrors([
                        'message' => 'Error de contraseña o usuario, comuniquese con el administrador del sistema.',
                    ]);
                }
            }
        }
        else{
            $attempt = Auth::attempt($credentials);
            if($attempt){
                return redirect()->intended('home');
            }
            else{
                return back()->withErrors([
                    'message' => 'No tiene acceso al sistema, comuniquese con el administrador del sistema.',
                ]);
            }
        }
    }
    
    public function refreshCaptcha()
    {
        try {
            $captcha = captcha_src('default');
            
            return response()->json([
                'captcha' => $captcha
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'error' => $e->getMessage()
            ], 500);
        }
        
    }

    public function logout(Request $request)
    {
        // Verifica si el usuario está autenticado
        if (Auth::check()) {
            Auth::logout();
            $request->session()->invalidate();
            $request->session()->regenerateToken();
            return redirect('/login')->with('message', 'Sesión cerrada correctamente.');
        }

        return response()->json(['message' => 'Session closed in system 2'], 200);
    }
}
