<?php

namespace App\Models;

use App\Models\CuerpoFormulario;
use Illuminate\Support\Facades\DB;
use Illuminate\Database\Eloquent\Model;
use OwenIt\Auditing\Contracts\Auditable;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class ParTipoformulario extends Model implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use HasFactory,SoftDeletes;

    protected $table = 'par_tipoformulario';
    protected $primaryKey = 'idpar_tipoformulario';

    public $timestamps = true;
    
    const CREATED_AT = 'fecha_creacion';
    const UPDATED_AT = 'fecha_modificacion';
    const DELETED_AT = 'fecha_eliminacion';
    
    const CREATED_BY = 'usuario_creacion';
    const UPDATED_BY = 'usuario_modificacion';
    const DELETED_BY = 'usuario_eliminacion';

    protected $fillable = [
        'nombre', 'logo', 'descripcion',
        'tipo', 'slug', 'fecha_inicio', 'fecha_fin',
        'usuario_creacion', 'fecha_creacion',
        'usuario_modificacion', 'fecha_modificacion',
        'usuario_eliminacion', 'fecha_eliminacion','estado','eliminado',
    ];
    
    //excluir o incluir los campos que deseas auditar
    protected $auditInclude = [
        'nombre', 'logo', 'descripcion',
        'tipo', 'slug', 'fecha_inicio', 'fecha_fin',
        'usuario_creacion', 'fecha_creacion',
        'usuario_modificacion', 'fecha_modificacion',
        'usuario_eliminacion', 'fecha_eliminacion','estado','eliminado',
    ];

    public static function boot()
    {
        parent::boot();
        
        // Ahora solo necesitamos manejar los campos de usuario
        static::creating(function ($model) {
            $model->{self::CREATED_BY} = auth()->check() ? auth()->user()->name : null;
            $model->{self::CREATED_AT} = now(); 
            // No necesitamos establecer CREATED_AT, Laravel lo hace automáticamente
        });
        
        static::updating(function ($model) {
            $model->{self::UPDATED_BY} = auth()->check() ? auth()->user()->name : null;
            $model->{self::UPDATED_AT} = now();
            // No necesitamos establecer UPDATED_AT, Laravel lo hace automáticamente
        });
        
        // Para la eliminación suave
        static::deleting(function ($model) {
            if (!$model->isForceDeleting()) {
                $model->{self::DELETED_BY} = auth()->check() ? auth()->user()->name : null;
                $model->eliminado = true;
                $model->save(); // Guardamos para que se apliquen estos cambios
                // DELETED_AT será manejado automáticamente por SoftDeletes
            }
        });
    }

    public function delete()
    {
        // Actualizamos los registros relacionados si no es una eliminación forzada
        if (!$this->isForceDeleting()) {
            $usuarioActual = auth()->check() ? auth()->user()->name : null;
            $fechaActual = now();
            
            // Actualizar planillas
            DB::table('rep_formularioplanilladato')
                ->where('idpar_tipoformulario', $this->idpar_tipoformulario)
                ->whereNull(self::DELETED_AT)
                ->update([
                    self::DELETED_BY => $usuarioActual,
                    self::DELETED_AT => $fechaActual,
                    'eliminado' => true
                ]);
            
            // Actualizar secciones
            DB::table('rep_formularioseccion')
                ->where('idpar_tipoformulario', $this->idpar_tipoformulario)
                ->whereNull(self::DELETED_AT)
                ->update([
                    self::DELETED_BY => $usuarioActual,
                    self::DELETED_AT => $fechaActual,
                    'eliminado' => true
                ]);
        }
        
        return parent::delete();
    }

    public function planillas()
    {
        return $this->hasMany(RepFormularioplanilladato::class, 'idpar_tipoformulario');
    }

    public function secciones()
    {
        return $this->hasMany(RepFormularioseccion::class, 'idpar_tipoformulario');
    }
}
