<?php

namespace App\Livewire\Procesos;

use App\Models\User;
use App\Models\Proceso;
use Livewire\Component;
use App\Models\Parametro;
use App\Models\ProcesoItem;
use Illuminate\Support\Str;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\DB;

class NuevoProcesoContratacionComponent extends Component
{
    use WithFileUploads;
    
    public $entidad_id;
    public $responsable_id;
    public $fecha_ingreso;
    public $gerencia_id;
    public $unidad_id;
    public $objeto_contratacion;
    public $normativa_id;
    public $modalidad_id;
    public $tipoContratacionId;
    public $formaAdjudicacionId;
    public $formalizacion;
    public $nuri_informe;
    public $nuri_nota_autorizacion;
    public $codigo_pac;
    public $metodo_seleccion_id;

    public $modalidades = [];
    public $contrato;
    public $items = [];
    public $tipo;
    public $procesoId;
    public $proceso;
    public $formasDisponibles = [];
    public $tiposContratacion = [];
    public $responsables=[];
    public $empresas;
    public $entidades = [];
    public $gerencias = [];
    public $unidades = [];
    public $normativas = [];
    public $formasPago = [];
    public $metodosSeleccion = [];
    public $presupuesto_total = 0;
    public $tiposFormalizacion = [];

    public $itemForm = [
        'nombre' => '',
        'monto' => '',
        'unidad' => '',
        'tiempo_entrega' => '',
        'forma_entrega' => '',
        'lugar_entrega' => '',
        'garantia' => null,
    ];

    protected function rulesProceso()
    {
        $reglas = [
            'entidad_id' => 'required|exists:parametros,id',
            'responsable_id' => 'nullable|exists:users,id',
            'fecha_ingreso' => 'required|date',
            'unidad_id' => 'required|exists:parametros,id',
            'objeto_contratacion' => 'required|string|max:255',
            'normativa_id' => 'required|exists:parametros,id',
            'modalidad_id' => 'required|exists:parametros,id',
            'tipoContratacionId' => 'required|exists:parametros,id',
            'formaAdjudicacionId' => 'required|exists:parametros,id',
            // 'formalizacion' => 'required',
            'nuri_informe' => 'required',
            'nuri_nota_autorizacion' => 'required',
            'items.*.nombre' => 'required|string',
            'items.*.monto' => 'required|numeric|min:0',
            'items.*.unidad' => 'required|string',
            'items.*.tiempo_entrega' => 'required|numeric',
            'items.*.forma_entrega' => 'required|string',
            'items.*.lugar_entrega' => 'required|string',
            'items.*.forma_pago' => 'required|string',
        ];

        if($this->modalidad_id == 22){
            $reglas['codigo_pac'] = 'required';
        }

        return $reglas;
    }

    protected function rulesItem()
    {
        $reglas = [
            'itemForm.nombre' => 'required|string',
            'itemForm.monto' => 'required|numeric|min:0',
            'itemForm.unidad' => 'required|string',
            'itemForm.tiempo_entrega' => 'required|numeric',
            'itemForm.forma_entrega' => 'required|string',
            'itemForm.lugar_entrega' => 'required|string',
            'itemForm.garantia' => 'nullable',
            'itemForm.forma_pago' => 'required',
        ];

        if ($this->itemForm['tiempo_entrega'] > 15) {
            $reglas['itemForm.garantia'] = 'required';
        }

        return $reglas;
    }

    public function getPacHabilitadoProperty()
    {
        return $this->modalidad_id == 22;
    }

    public function getMetodoSeleccionHabilitadoProperty()
    {
        return $this->modalidad_id == 23;
    }

    public function updatedModalidadId($value)
    {
        if ($value != 22) {
            $this->codigo_pac = null;
        }
        if ($value != 23) {
            $this->metodo_seleccion_id = null;
        }
    }

    public function mount($procesoId=null)
    {
        $this->fecha_ingreso = date('Y-m-d');
        $this->normativas = Parametro::where('tipo','normativa')->get();
        $this->formasPago = Parametro::where('tipo','forma_pago')->get();
        $this->modalidades = Parametro::where('tipo','modalidad')->get();
        $this->tiposContratacion = Parametro::where('tipo','tipo_contratacion')->get();
        $this->tiposFormalizacion = Parametro::where('tipo','tipo_formalizacion')->get();
        $this->metodosSeleccion = Parametro::where('tipo','metodo_seleccion')->get();
        $this->responsables = User::all();
        $this->entidades = Parametro::where('tipo','empresa')->get();
        $this->tipoContratacionId = old('tipo_contratacion_id');
        $this->gerencias = DB::connection('pgsql_intranet')
            ->table('unidads')
            ->select('id', 'nombre')->where('tipo', 1)->orderBy('orden')
            ->get();
        $this->unidades = collect();

        if ($this->tipoContratacionId) {
            $this->updatedTipoContratacionId();
        }
        
        if ($procesoId) {
            $this->proceso = Proceso::findOrFail($procesoId);
            $this->procesoId = $procesoId;
            $procesoItem = ProcesoItem::with('items')->findOrFail($procesoId);
            $this->tipo = $procesoItem->tipo;
            $this->items = $procesoItem->items->map(fn($it) => [
                'id' => $it->id,
                'nombre' => $it->nombre,
                'monto' => $it->monto,
                'unidad_medida' => $it->unidad_medida,
            ])->toArray();
        }
    }

    public function updatedGerenciaId($value)
    {
        $this->unidades = DB::connection('pgsql_intranet')
            ->table('unidads')
            ->select('id', 'nombre')->where('tipo', 2)->where('area_id', $value)
            ->get();
        $this->unidad_id = null; // reset unidad
        $this->dispatch('refreshSelect2'); // evento para actualizar select2 en JS
    }

    public function updatedTipoContratacionId()
    {
        $this->formasDisponibles = Parametro::formasAdjudicacionDe($this->tipoContratacionId)->get();
        $this->formaAdjudicacionId = null; // resetear valor si cambia tipo
    }
    public function addItem()
    {
        $this->items[] = [
            'nombre' => '',
            'monto' => 0,
            'unidad_medida' => 'Pieza',
        ];
    }

    public function render()
    {
        return view('livewire.procesos.nuevo-proceso-contratacion-component');
    }

    public $showItemModal = false;
    public $editingIndex = null; 

    public function openItemModal($index = null): void
    {
        $this->resetErrorBag();
        $this->editingIndex = $index;

        if (!is_null($index)) {
            $this->itemForm = $this->items[$index];
        } else {
            $this->itemForm = [
                'nombre' => '',
                'monto' => '',
                'unidad' => '',
                'tiempo_entrega' => '',
                'forma_entrega' => '',
                'lugar_entrega' => '',
                'forma_pago' => '',
                'garantia' => null,
            ];
        }

        $this->showItemModal = true;
    }

    public function saveItem(): void
    {
        $this->validate($this->rulesItem(),[],[],'items');

        if (!is_null($this->editingIndex)) {
            $this->items[$this->editingIndex] = $this->itemForm;
        } else {
            $this->items[] = $this->itemForm;
        }

        $this->presupuesto_total = array_sum(array_column($this->items, 'monto'));
        
        $this->showItemModal = false;
        $this->resetErrorBag('items');
        $this->dispatch('cerrar-modal'); // para el frontend
    }

    public function removeItem(int $index): void
    {
        unset($this->items[$index]);
        $this->presupuesto_total = array_sum(array_column($this->items, 'monto'));
        $this->items = array_values($this->items); // reindexar
    }

    public function guardarProceso()
    {
        $this->validate($this->rulesProceso(),[],[],'proceso');

        // Guardar contrato
        $contrato = new Proceso();
        $contrato->nuri_informe = $this->nuri_informe;
        $contrato->nuri_nota_autorizacion = $this->nuri_nota_autorizacion;
        $contrato->entidad_id = $this->entidad_id;
        $contrato->fecha_ingreso = $this->fecha_ingreso;
        $contrato->unidad_solicitante = $this->unidad_id;
        $contrato->objeto_contratacion = $this->objeto_contratacion;
        $contrato->normativa_id = $this->normativa_id;
        $contrato->modalidad_id = $this->modalidad_id;
        $contrato->tipo_contratacion_id = $this->tipoContratacionId;
        $contrato->forma_adjudicacion_id = $this->formaAdjudicacionId;
        $contrato->tipo_formalizacion = $this->formalizacion;
        $contrato->codigo_pac= $this->codigo_pac;
        $contrato->save();
        
        if ($this->responsable_id != null) {
            $contrato->asignaciones()->syncWithoutDetaching([$this->responsable_id => ['fecha_asignacion' => now(),'observacion' => 'Asignación inicial']]);
        }
        $totalMonto = 0;
        // Guardar items
        foreach ($this->items as $item) {
            $nuevoItem = new ProcesoItem();
            $nuevoItem->proceso_id = $contrato->id;
            $nuevoItem->nombre = $item['nombre'];
            $nuevoItem->monto = $item['monto'];
            $nuevoItem->unidad_medida = $item['unidad'];
            $nuevoItem->tiempo_entrega = $item['tiempo_entrega'];
            $nuevoItem->forma_entrega = $item['forma_entrega'];
            $nuevoItem->lugar_entrega = $item['lugar_entrega'];
            $nuevoItem->garantia_entrega = $item['garantia'];
            $nuevoItem->forma_pago = $item['forma_pago'];
            $nuevoItem->garantia_entrega = $item['garantia'];
            $nuevoItem->save();
            $totalMonto += $item['monto'];
        }
        //dd($totalMonto);
        $contrato->precio_referencial = $totalMonto;
        $contrato->save();
        
        $this->dispatch('alerta', text:'Proceso de Contratación registrado correctamente');
        $this->reset();
        return redirect()->route('procesos-contratacion.index');
    }
    
}
